#include "helper.h"
#include "init.h"



int read_parameters( const char *szFilename,     
                    double *Re,                
                    double *UI,                /* velocity in  x-direction */
                    double *VI,                /* velocity y-direction */
                    double *PI,                /* pressure */
                    double *GX,                /* Gravitation x-direction */
                    double *GY,                /* Gravitation y-direction */
                    double *t_end,             /* end time */
                    double *xlength,           /* domain length x-direction */
                    double *ylength,           /* domain lenght y-direction */
                    double *dt,                /* time step */
                    double *dx,                /* cell length x-direction*/
                    double *dy,                /* cell length y-direction */
                    int  *imax,              /* number of cells in x- and y-directions */
                    int  *jmax,             
                    double *alpha,             /* Uppwind-Differencing-Factor alpha */
                    double *omg,               /* Relaxation factor Omega */
                    double *tau,               /* safety parameter for time step calculation */
                    int  *itermax,           /* max. number of pressure iterations  */
                    double *eps,               /* tolerance limit for pressure calculation */
		    double *dt_value,          /* time steps for output */
		    char *problem,           /* problem name */
		    int *wl,                 /*boundary value flags, links rechts top bottom*/
		    int *wr,
		    int *wt,
		    int *wb,
		    double *dp,             /* pressure difference */
		    int *N,                 /*number of particles on the line (x1,y1)_(x2,y2) */
		    double *x1,
		    double *y1,
		    double *x2,
		    double *y2,
		    double *dt_path,        /*time step for the output of pathlines */
		    double *dt_streak,     /*time step for the output of streaklines */
		    double *dt_insert)    /*time step for the injection of particles */
{
   READ_DOUBLE( szFilename, *xlength );
   READ_DOUBLE( szFilename, *ylength );

   READ_DOUBLE( szFilename, *Re    );
   READ_DOUBLE( szFilename, *t_end );
   READ_DOUBLE( szFilename, *dt    );

   READ_INT   ( szFilename, *imax );
   READ_INT   ( szFilename, *jmax );

   READ_DOUBLE( szFilename, *omg   );
   READ_DOUBLE( szFilename, *eps   );
   READ_DOUBLE( szFilename, *tau   );
   READ_DOUBLE( szFilename, *alpha );

   READ_INT   ( szFilename, *itermax );
   READ_DOUBLE( szFilename, *dt_value );

   READ_DOUBLE( szFilename, *UI );
   READ_DOUBLE( szFilename, *VI );
   READ_DOUBLE( szFilename, *GX );
   READ_DOUBLE( szFilename, *GY );
   READ_DOUBLE( szFilename, *PI );

   READ_STRING( szFilename, problem);
   READ_INT   ( szFilename, *wl);
   READ_INT   ( szFilename, *wr);
   READ_INT   ( szFilename, *wt);
   READ_INT   ( szFilename, *wb);
   READ_DOUBLE( szFilename, *dp );

   *dx = *xlength / (double)(*imax);
   *dy = *ylength / (double)(*jmax);

   READ_INT   ( szFilename, *N  );
   READ_DOUBLE( szFilename, *x1 );
   READ_DOUBLE( szFilename, *y1 );
   READ_DOUBLE( szFilename, *x2 );
   READ_DOUBLE( szFilename, *y2 );
   READ_DOUBLE( szFilename, *dt_path );
   READ_DOUBLE( szFilename, *dt_streak );
   READ_DOUBLE( szFilename, *dt_insert );

   return 1;
}

void init_uvp(double UI,
	      double VI,
	      double PI,
	      int imax,
	      int jmax,
	      double **U,
	      double **V,
	      double **P)
{
  int i,j;
  for(i=0;i<=imax;i++){
    for(j=0;j<=jmax+1;j++){
      U[i][j] = UI;
    }
  }
  for(i=0;i<=imax+1;i++){
    for(j=0;j<=jmax;j++){
      V[i][j] = VI;
    }
  }
  for(i=0;i<=imax+1;i++){
    for(j=0;j<=jmax+1;j++){
      P[i][j] = PI;
    }
  }
}

void init_flag(char *problem,
	       int imax,
	       int jmax,
	       int **Flag)
{
  int i,j,m,n,i0,j0;

  /*
   * initialize the boundary cells
   */
  for(i=0;i<=imax+1;i++){
    for(j=0;j<=jmax+1;j++){
      if(i>=1 && i<=imax && j>=1 && j<=jmax)
	Flag[i][j] = C_F;
      else
	Flag[i][j] = C_B;
    }
  }

  /*
   * insert problem dependent obstacle cells
   */
  if( strcmp(problem,"step") == 0 ){
    for(i=1;i<=jmax/2;i++){
      for(j=1;j<=jmax/2;j++){
	Flag[i][j] = C_B;
      }
    }
  }

  else if( strcmp(problem,"karman") == 0 ){
    m = jmax/5;
    printf("m = %d\n",m);
    for(i=2*m;i<=3*m  ;i++) Flag[i][i]       = C_B;
    for(i=2*m;i<=3*m+1;i++) Flag[i][i-1]     = C_B;
    for(i=2*m;i<=3*m  ;i++) Flag[i+1][i-1]   = C_B;
  }

  else if( strcmp(problem,"cicrle") == 0 ){
    m = jmax/2;
    n = jmax/3;
    printf("m = %d\n",m);
    printf("n = %d\n",n);
    for(i=1;i<=imax;i++){
      for(j=1;j<=jmax;j++){
	if(  (i-2*m)*(i-2*m) + (j-m)*(j-m) < n*n ) {
	  Flag[i][j] = C_B; 
//	  Flag[i+1][j] = C_B; 
//	  Flag[i][j+1] = C_B; 
//	  Flag[i+1][j+1] = C_B; 
	}
      }
    }
  }

  /*
  else if( strcmp(problem,"Kanal") == 0 ){

    for(i=1;i<=imax;i++){
      Flag[i][jmax  ] = C_B;
      Flag[i][jmax-1] = C_B;
      Flag[i][jmax-2] = C_B;
    }

    for(j=1;j<=jmax;j++){
      Flag[imax][  j] = C_B;
      Flag[imax-1][j] = C_B;
      Flag[imax-2][j] = C_B;
    }
  }
  */

  else if( strcmp(problem,"quadrat") == 0 ){
    for(i=4*jmax/8;i<=5*jmax/8;i++){
      for(j=4*jmax/8;j<=5*jmax/8;j++){
	Flag[i][j] = C_B;
      }
    }
  }

  else if( strcmp(problem,"Tragflaeche") == 0 ){
    for(i=1;i<=imax;i++){
      for(j=1;j<=jmax;j++){
	if(  j>jmax/2 &&  j<  jmax*(0.6 - (0.3/imax)*(i-imax/5)*(i-imax/5))  ) {
	  Flag[i][j] = C_B; 
	  Flag[i+1][j] = C_B; 
	  Flag[i][j+1] = C_B; 
	  Flag[i+1][j+1] = C_B; 
	}
      }
    }
  }


  else if( strcmp(problem,"See") == 0 ){
    j0 = (jmax)/2+1;
    i0 = (imax)/2+1;
    printf("m = %d\n",m);
    printf("n = %d\n",n);
    for(j=1;j<=jmax;j++){
      for(i=1;i<=imax;i++){
	if(  (float)((i-i0)*(i-i0))/(float)(i0*i0) + (float)(j-jmax)*(j-jmax)/(float)(jmax*jmax) > 1.0 ) {
          Flag[i][j] = C_B; 
	}
	else{
	}
      }
    }
  }

  /*
   * set another flags
   */

  for(i=0;i<=imax+1;i++){
    for(j=0;j<=jmax+1;j++){
      if(i>1      && ((Flag[i-1][j]&C_TYP) == C_F)) Flag[i][j] |= C_W;
      if(i<imax+1 && ((Flag[i+1][j]&C_TYP) == C_F)) Flag[i][j] |= C_O;
      if(j>1      && ((Flag[i][j-1]&C_TYP) == C_F)) Flag[i][j] |= C_S;
      if(j<jmax+1 && ((Flag[i][j+1]&C_TYP) == C_F)) Flag[i][j] |= C_N;
    }
  }

  /*
   * test forbidden cells
   */


  m=0;
  for(j=jmax+1;j>=0;j--){
    for(i=0;i<=imax+1;i++){
      if((Flag[i][j]&C_TYP) == C_B)      /* obstacle cell */
	if((Flag[i][j]&C_H) != C_N   &&  /* not edge cell */
	   (Flag[i][j]&C_H) != C_S   && 
	   (Flag[i][j]&C_H) != C_W   &&
	   (Flag[i][j]&C_H) != C_O   &&
	   (Flag[i][j]&C_H) != C_NO  &&  /* not corner cell */
	   (Flag[i][j]&C_H) != C_SO  &&
	   (Flag[i][j]&C_H) != C_NW  &&
	   (Flag[i][j]&C_H) != C_SW  &&
	   (Flag[i][j]&C_H) != 0         
	   ){    
	  m++;
	  //printf("OO");
	}
//	else printf("##");
//      else printf("  ");
    }
//    printf("\n");
  }
  if(m>0){
    printf("Attention: %d forbidden obstacle cells was found ! \n",m);
    exit(0);
  }

  if( strcmp(problem,"See") == 0 || strcmp(problem,"Stufe") == 0 || strcmp(problem,"plane") == 0){
    for (j=1; j<=jmax;j++){
      if ( Flag[0][j]&C_O      ) Flag[0][j]      |= C_dW;
      if ( Flag[imax+1][j]&C_W ) Flag[imax+1][j] |= C_dO;
    }
  }


  for(j=jmax+1;j>=0;j--){
    for(i=0;i<=imax+1;i++){
      printf ("%2d ",Flag[i][j]);
    }
    printf ("\n");
  }
}
