#include "helper.h"
#include "visual.h"
#include <stdio.h>

void output_uvp(const char *szProblem,
		 double xlength,
                 double ylength,
                 int  imax,
                 int  jmax,
		 double dx,
		 double dy,
                 double **U,
                 double **V,
                 double **P,
		 int **Flag)
{
    static int nCount = 0;  /* counts the function calls */
    int i, j;
    double **tmp = matrix( 1, imax, 1, jmax );    /* temp. matrix */
    double **Psi;
    double **Zeta;
    char szFileName[80];            /* file name string */
    
    printf("output is written\n");
    if ( nCount==0 ){               /* coordinates for AVS: only once */
      for( i = 1; i <= imax; i++){
        for( j = 1; j <= jmax; j++){
          tmp[i][j] = ((float)i-0.5)*dx;
        }
      }
      sprintf( szFileName, "%s.X.koo", szProblem );
      write_matrix( szFileName,tmp,1,imax,1,jmax,xlength,ylength,nCount==0);
      for( i = 1; i <= imax; i++){
        for( j = 1; j <= jmax; j++){
          tmp[i][j] = ((float)j-0.5)*dy;
        }
      }
      sprintf( szFileName, "%s.Y.koo", szProblem );
      write_matrix( szFileName,tmp,1,imax,1,jmax,xlength,ylength,nCount==0);
    }

    /* U-Matrix */
    for( i = 1; i <= imax; i++){
      for( j = 1; j <= jmax; j++){
        if ( Flag[i][j]&C_F ) tmp[i][j] = (U[i-1][j] + U[i][j]) * 0.5;
        else tmp[i][j] = 0.;
      }
    }

    sprintf( szFileName, "%s.U.out", szProblem );

    write_matrix( szFileName,tmp,1,imax,1,jmax,xlength,ylength,nCount==0);

    /* V-Matrix */
    for( i = 1; i <= imax; i++){
      for( j = 1; j <= jmax; j++){
        if ( Flag[i][j]&C_F ) tmp[i][j]  = (V[i][j-1] + V[i][j]) * 0.5;
        else tmp[i][j] = 0.;
      }
    }

    sprintf( szFileName, "%s.V.out", szProblem );

    write_matrix( szFileName,tmp,1,imax,1,jmax,xlength,ylength,nCount==0);

    /* P-Matrix */
    sprintf( szFileName, "%s.P.out", szProblem );

    write_matrix( szFileName,P,1,imax,1,jmax,xlength,ylength,nCount==0);

    free_matrix( tmp,1,imax,1,jmax ); 

    /* write field file for AVS */
    write_fieldfile( szProblem,imax,jmax,xlength,ylength,nCount+1);

    Psi = matrix(0,imax,0,jmax);
    Zeta = matrix(1,imax,1,jmax);

    calculate_psi_zeta(dx,dy,imax,jmax,U,V,Psi,Zeta);

    sprintf( szFileName, "%s.psi.out", szProblem );
    write_matrix( szFileName,Psi,1,imax,1,jmax,xlength,ylength,nCount==0);

    sprintf( szFileName, "%s.zeta.out", szProblem );
    write_matrix( szFileName,Zeta,1,imax,1,jmax,xlength,ylength,nCount==0);

    free_matrix( Zeta, 1, imax, 1, jmax );
    free_matrix( Psi, 0, imax, 0, jmax );

    ++nCount;
}

void calculate_psi_zeta(double dx,
		       double dy,
		       int imax,
		       int jmax,
		       double **U,
		       double **V,
		       double **Psi,
		       double **Zeta )
{

  int i,j;
  for(i=0;i<=imax;i++){
    Psi[i][0] = (i==0) ? 0.0 : Psi[i-1][0] - V[i][0]*dx;
    for(j=1;j<=jmax;j++){
      Psi[i][j] = Psi[i][j-1] + U[i][j]*dy;
    }
  }
  for(i=1;i<imax;i++){
    for(j=1;j<jmax;j++){
      Zeta[i][j] = (U[i][j+1]-U[i][j])/dy  -  (V[i+1][j]-V[i][j])/dx;
    }
  }

}


void write_fieldfile( const char *szProblem, int imax, int jmax, 
                      double xlength, double ylength, int nCount ){

  char szFileName[80];	   
  FILE *fp=NULL;

  sprintf( szFileName, "%s.fld", szProblem );

  fp = fopen( szFileName, "w");
  fprintf(fp,"# AVS field file %s\n",szProblem);
  fprintf(fp,"#\n");
  fprintf(fp,"nstep   = %d\n",nCount);
  fprintf(fp,"ndim    = 2\n");
  fprintf(fp,"dim1    = %d\n",imax);
  fprintf(fp,"dim2    = %d\n",jmax);
  fprintf(fp,"nspace  = 2\n");
  fprintf(fp,"veclen  = 5\n");
  fprintf(fp,"data    = float\n");
  fprintf(fp,"field   = irregular\n");
  fprintf(fp,"min_ext = %f %f\n",0.0,0.0);
  fprintf(fp,"max_ext = %f %f\n",xlength,ylength);
  sprintf( szFileName, "%s.X.koo", szProblem );
  fprintf(fp,"coord 1 file=%s filetype=binary skip=0 offset=0 stride=1 close=1\n",szFileName);
  sprintf( szFileName, "%s.Y.koo", szProblem );
  fprintf(fp,"coord 2 file=%s filetype=binary skip=0 offset=1 stride=1 close=1\n",szFileName);
  fprintf(fp,"DO\n");
  sprintf( szFileName, "%s.U.out", szProblem );
  fprintf(fp,"variable 1 file=%s filetype=binary skip=0 offset=0 stride=1 close=0\n",szFileName);
  sprintf( szFileName, "%s.V.out", szProblem );
  fprintf(fp,"variable 2 file=%s filetype=binary skip=0 offset=0 stride=1 close=0\n",szFileName);
  sprintf( szFileName, "%s.P.out", szProblem );
  fprintf(fp,"variable 3 file=%s filetype=binary skip=0 offset=0 stride=1 close=0\n",szFileName);
  sprintf( szFileName, "%s.psi.out", szProblem );
  fprintf(fp,"variable 4 file=%s filetype=binary skip=0 offset=0 stride=1 close=0\n",szFileName);
  sprintf( szFileName, "%s.zeta.out", szProblem );
  fprintf(fp,"variable 5 file=%s filetype=binary skip=0 offset=0 stride=1 close=0\n",szFileName);
  fprintf(fp,"EOT\n");
  fprintf(fp,"ENDDO\n");

  fclose(fp);
}


struct particle_line *set_particle(int N, 
				      double x1, 
				      double y1, 
				      double x2, 
				      double y2,
				      double xlength,
				      double ylength,
				      char* Outputfile)
{
  int k;
  struct particle_line *pl;

  pl = (struct particle_line*)malloc(N*sizeof(struct particle_line));
  for(k=0;k<N;k++){
    pl[k].Particle = (struct particle*)malloc(sizeof(struct particle));
    pl[k].Particle->x = x1 + (double)k/(N-1)*(x2-x1);
    pl[k].Particle->y = y1 + (double)k/(N-1)*(y2-y1);
    pl[k].Particle->next = NULL;
    pl[k].length = 0;
  }

  return pl;
}



void move_particle(struct particle_line *pl, 
		     int N,
		     double dt, 
		     double dx, 
		     double dy, 
		     int imax, 
		     int jmax, 
		     double **U, 
		     double **V,
		     int **Flag){

  int i,j,k;
  struct particle *p;
  struct particle *pn;
  double x,y;
  double x1,x2,y1,y2;
  double u1,u2,u3,u4,u;
  double v1,v2,v3,v4,v;
  for(k=0;k<N;k++)
    if(pl[k].length > 0){
      p = pl[k].Particle;
      while(p != NULL && p->next != NULL){
        pn = p->next;
	x = pn->x;
	y = pn->y;
	i = (int)(x/dx)+1;
	j = (int)((y+dy/2)/dy)+1;
	x1 = (i-1)*dx;
	x2 = i*dx;
	y1 = ((j-1)-0.5)*dy;
	y2 = (j-0.5)*dy;
	u1 = U[i-1][j-1];
	u2 = U[i][j-1];
	u3 = U[i-1][j];
	u4 = U[i][j];
	u = (1.0/dx/dy)*(  (x2-x)*(y2-y)*u1 + (x-x1)*(y2-y)*u2 +
			   (x2-x)*(y-y1)*u3 + (x-x1)*(y-y1)*u4   );
	
	i = (int)((x+dx/2)/dx)+1;
	j = (int)(y/dy)+1;
	x1 = ((i-1)-0.5)*dx;
	x2 = (i-0.5)*dx;
	y1 = (j-1)*dy;
	y2 = j*dy;
	v1 = V[i-1][j-1];
	v2 = V[i][j-1];
	v3 = V[i-1][j];
	v4 = V[i][j];
	v = (1.0/dx/dy)*(  (x2-x)*(y2-y)*v1 + (x-x1)*(y2-y)*v2 +
			   (x2-x)*(y-y1)*v3 + (x-x1)*(y-y1)*v4   );
	
	pn->x += u*dt;
	pn->y += v*dt;
	i = (int)(x/dx)+1;
	j = (int)(y/dy)+1;

        if(pn->x < 0 || pn->y < 0 || pn->x > imax*dx || pn->y > jmax*dy || (Flag[i][j]&C_TYP) != C_F){
          p->next = pn->next;
	  free(pn);
          pl[k].length--;
	}

        p = p->next;
      }
    }    
}

void insert_particle(struct particle_line *pl, 
			int N)
{
  int k;
  struct particle *erstp;
  struct particle *neup;
  struct particle *tmpp;
  for(k=0; k<N; k++){
    erstp = pl[k].Particle;
    neup = (struct particle*)malloc(sizeof(struct particle));
    neup->x = erstp->x; // copy positions of the first particle
    neup->y = erstp->y;
    tmpp = erstp->next; // replace after the first particle
    erstp->next = neup;
    neup->next = tmpp;
    pl[k].length++;
  }
}


void write_particle(struct particle_line *pl, 
		       int N, 
		       int partmax, 
		       char* outputfile)
{
  float x,y;
  int anzpl;
  int npart;
  int k;
  struct particle *p;

  FILE * fh = 0;

  fh = fopen( outputfile ,"a");	/* append to the file*/
  if( fh == NULL )		       
  {
    char szBuff[80];
    sprintf( szBuff, "Failed to open %s", outputfile );
    ERROR( szBuff );
  }
  
  anzpl = 0;
  for( k=0; k<N; k++) anzpl += pl[k].length; // set the number of particles
//  fprintf(fh,"%d\n",anzpl);
//  printf("number of particles = %d\n",anzpl);
  
  npart=0;

  for( k=0; k<N; k++){         // write all the particle position written in the output file
    if(pl[k].length>0){
      p = (pl[k].Particle)->next;
      while(p){
        x = p->x < 1e-20 ? 0 : p->x;
        y = p->y < 1e-20 ? 0 : p->y;

	fprintf(fh,"%f\t%f\n",x,y);
        npart++;
	p = p->next;
      }
    }
  }

  while ( npart<partmax ){
    fprintf(fh,"0.0 0.0\n");
    npart++;
  }
  if( fclose(fh) )
  {
    char szBuff[80];
    sprintf( szBuff, "Failed to close %s", outputfile );
    ERROR( szBuff );
  }
}


void pathlines(struct particle_line *pl, 
		int N, 
		double dt_path, 
		double t, 
		double dx, 
		double dy, 
		double dt, 
		int imax, 
		int jmax, 
		double **U, 
		double **V,
		int **Flag,
		char* outputfile){

  static int nCount=1;
  char szFileName[80];

  if ( nCount==1 ){
    sprintf( szFileName, "rm %s.prt\n", outputfile);
    system(szFileName);
  }

  if( t==0.0 ){
    insert_particle(pl,N);
  }

  move_particle(pl, N, dt, dx, dy, imax, jmax, U, V, Flag);

  if(  t - ((int)(t/dt_path))*dt_path <= dt ){
    sprintf( szFileName, "%s.prt", outputfile);
    write_particle(pl,N,0,szFileName);
    write_particlefieldfile(outputfile,N*nCount,1);
    nCount++;
  }
}


void streaklines(struct particle_line *pl, 
		   int N, 
		   double dt_replace, 
		   double dt_streak, 
		   double t, 
		   double dx, 
		   double dy, 
		   double dt, 
		   int imax, 
		   int jmax, 
		   double **U, 
		   double **V, 
		   int **Flag,
		   char* outputfile){

  static int nCount=1;
  char szFileName[80];
  int partmax = 10000;

  if ( nCount==1 ){
    sprintf( szFileName, "rm %s.prt\n", outputfile);
    system(szFileName);
  }

  if(  t - ((int)(t/dt_replace))*dt_replace <= dt ){
    insert_particle(pl,N);
  }
  move_particle(pl,N,dt,dx,dy,imax,jmax,U,V,Flag);      

  if(  t - ((int)(t/dt_streak))*dt_streak <= dt ){
    sprintf( szFileName, "%s.prt", outputfile);
    write_particle(pl,N,partmax,szFileName);
    write_particlefieldfile(outputfile,partmax,nCount);
    nCount++;
  }
}


void
write_particlefieldfile(char *outputfile, int partmax, int nCount){

  FILE *fp=NULL;
  char szFileName[80];

  sprintf( szFileName, "%s.fld", outputfile);

  fp = fopen( szFileName,"w");
  fprintf(fp,"# AVS Particle-Fieldfile %s\n",outputfile);
  fprintf(fp,"#\n");
  fprintf(fp,"nstep=%d\n",nCount);
  fprintf(fp,"ndim=2\n");
  fprintf(fp,"dim1=%d\n",partmax-1);
  fprintf(fp,"dim2=1\n");
  fprintf(fp,"nspace=2\n");
  fprintf(fp,"veclen=0\n");
  fprintf(fp,"data=float\n");
  fprintf(fp,"field=irregular\n");
  fprintf(fp,"DO\n");
  sprintf( szFileName, "%s.prt", outputfile);
  fprintf(fp,"coord 1 file=%s filetype=ascii skip=0 offset=0 stride=2 close=0\n",szFileName);
  fprintf(fp,"coord 2 file=%s filetype=ascii skip=0 offset=1 stride=2 close=0\n",szFileName);
  fprintf(fp,"EOT\n");
  fprintf(fp,"ENDDO\n");

  fclose(fp);

  ++nCount;
}
